/* gopher.c
 *
 * Part of the Internet Gopher program, copyright (C) 1991, 1992
 * University of Minnesota Microcomputer Workstation and Networks Center
 *
 * See the README file for information about the Gopher program.
 */
#pragma comment (exestr, "Built on " __DATE__ " at " __TIME__)

#include "gopher.h"
#ifdef SPAWNO
#include "spawno.h"
#endif
#ifdef LWP
extern struct hostent *lwpgethostbyname(char *);
#endif

extern unsigned _stklen = 543210U;

int NEAR describe_gopher(GopherStruct *);

/* Connect_to_gopher performs a connection to socket 'service' on host
 * 'host'.  Host can be a hostname or ip-address.  If 'host' is null, the
 * local host is assumed.   The parameter full_hostname will, on return,
 * contain the expanded hostname (if possible).  Note that full_hostname is a
 * pointer to a char *, and is allocated by connect_to_gopher()
 *
 * Errors:
 *
 * -1 get service failed
 *
 * -2 get host failed
 *
 * -3 socket call failed
 *
 * -4 connect call failed
 */

static struct sockaddr_in Server;
static char sBuf[16];
struct hostent *HostPtr;

int connect_to_gopher(Hostname, ThePort)
  char *Hostname;
  int  ThePort;
{
     int iSock = 0;

     if (Hostname == '\0') {
	  gethostname(sBuf, 16);
	  Hostname = sBuf;
     }

     if (HostPtr = gethostbyname(Hostname)) {
	  bzero((char *) &Server, sizeof(Server));
	  bcopy(HostPtr->h_addr, (char *) &Server.sin_addr, HostPtr->h_length);
	  Server.sin_family = HostPtr->h_addrtype;
     } else {
	if ((Server.sin_addr.s_addr = inet_addr(Hostname)) == -1) {
	       return (-2);
	} else {
	  Server.sin_family = AF_INET;
	}
     }

     Server.sin_port = htons(ThePort);

     if ((iSock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
	  return (-3);
/*
     setsockopt(iSock, SOL_SOCKET, SO_DONTLINGER, 0, 0);
     setsockopt(iSock, SOL_SOCKET, SO_REUSEADDR, 0, 0);
     setsockopt(iSock, SOL_SOCKET, SO_KEEPALIVE, 0, 0);
*/
     if (connect(iSock, (struct sockaddr *) &Server, sizeof(Server)) < 0) {
	  socketclose(iSock);
	  return (-4);
     }
     return(iSock);
}


/*
** Open a connection to another host
*/

int
NEAR do_telnet(ZeGopher)
  GopherStruct *ZeGopher;
{
     char sMessage1[128];
     char sMessage2[128];
     char sTelCmd[128]; 
     char ch;

     /* retrieve the gopher information for the telnet command*/

     clear();
     Centerline("Warning!!!!!, you are about to leave the Internet", 1);
     Centerline("Gopher program and connect to another host.", 2);
     Centerline("If you get stuck press the control key and the ] key,",3);
     Centerline("and then type quit",4);
     
     sprintf(sMessage1,"Now connecting to %s", GSgetHost(ZeGopher));
     if (*GSgetPath(ZeGopher) != '\0')
	  sprintf(sMessage2,"Use the account name \"%s\" to log in",
		  GSgetPath(ZeGopher));
     else
	  sMessage2[0] = '\0';

     Centerline(sMessage1, LINES/2 -1);
     Centerline(sMessage2, LINES/2 +1);
     Centerline("Press return to connect, q to cancel: ", LINES - 1);
     refresh();

     while ((ch = getch()) != '\r')
	if (ch == 'q')
		return(0);

     exit_curses();

     if(GSgetType(ZeGopher) == 'T')
	sprintf(sTelCmd, "%s %s", TN3270Command, GSgetHost(ZeGopher));
     else {
	if (ZeGopher->iPort != 0) 
             sprintf(sTelCmd, "%s %s %d",
           	TelnetCommand, GSgetHost(ZeGopher), GSgetPort(ZeGopher)); 
        else 
             sprintf(sTelCmd, "%s %s", TelnetCommand, GSgetHost(ZeGopher));
     }

     exit_curses();
     system(sTelCmd);
     init_curses();
     return(0);
}


/*
** do_index gets keywords from the user to search for.  It returns
** it to the calling process.  This storage is volotile. Callers should
** make a copy if they want to call do_index multiple times.
*/

char*
NEAR do_index(ZeGopher)
  GopherStruct *ZeGopher;
{
     static char inputline[WHOLELINE];

     GetOneOption("Index word(s) to search for: ", inputline);
     if (strlen(inputline) == 0)
	  return(NULL);
     else
	  return(inputline);
}

#define BUFSIZE 1400  /* A pretty good value for ethernet */
#ifdef IS_MESS_DOS   /* DOS machines can't fork to do sound */

int
NEAR do_sound(ZeGopher)
  GopherStruct *ZeGopher;
{
     int iLength, sockfd, filefd, cc;
     unsigned long int i = 0;
     char tmpfilename[_MAX_PATH];
     char inputline[512];
     char buf[BUFSIZE];

     if (PlayCommand[0] == '\0') {
	  /*** Hey! no play command, bummer ***/
	  CursesErrorMsg("Sorry, the sound player command is not set.");
	  return(1);
     }
     sprintf(tmpfilename, "%sgof%d.voc",tmpDirPath, getpid());

     /* open the file - if it already exists, we're about to clobber it */
     if ( (filefd = open( tmpfilename, O_WRONLY|O_TRUNC|O_CREAT|O_BINARY, S_IWRITE )) < 0 ) {
        /* should give more info here */
        Draw_Status( "Could not open file" );
        return(1);
     }

     Draw_Status("Connecting to server...");
     refresh();
     if ((sockfd = connect_to_gopher(GSgetHost(ZeGopher), GSgetPort(ZeGopher))) <0) {
	  check_sock(sockfd, GSgetHost(ZeGopher));
          close(filefd);
	  return(1);
     }

     /** Send out the request **/

     writestring(sockfd, GSgetPath(ZeGopher));
     writestring(sockfd, "\r\n");

     Draw_Status("Receiving file...");
     refresh();
     /* read and read until we get EOF */
     twirl();
     while ( (cc = recv( sockfd, buf, sizeof(buf), 0 )) > 0 ) {
        /* hope we don't have error while writing - I'm so lazy */
        write( filefd, buf, cc );
        twirl();
     }

     close(filefd); /* should error check */
     if ( cc < 0 )
        Draw_Status( "Error in file" );
     else
        Draw_Status( "File received ok" );
     socketclose( sockfd );
     sprintf(buf, "%s %s", PlayCommand, tmpfilename);
     system(buf);
     if (unlink(tmpfilename)!=0)
	  fprintf(stderr, "Couldn't unlink!!!\n"), exit(-1);
     return(0);
}

int
NEAR do_gif(ZeGopher)
  GopherStruct *ZeGopher;

{
     static char command[MAXSTR];
     char *p, Filename[_MAX_PATH], buf[BUFSIZE];   /* hope this is long enough :-) */
     int cc, sockfd, filefd;

     if (GifCommand[0] == '\0') {
	  /*** Hey! no GIF command, bummer ***/
	  CursesErrorMsg("Sorry, the GIF viewer command is not set.");
	  return(1);
     }

     sprintf(Filename, "%sgof%d.gif",tmpDirPath, getpid());

     /* open the file - if it already exists, we're about to clobber it */
     if ( (filefd = open( Filename, O_WRONLY|O_TRUNC|O_CREAT|O_BINARY, S_IWRITE )) < 0 ) {
        /* should give more info here */
        Draw_Status( "Could not open file" );
        return(1);
     }

     Draw_Status("Connecting to server...");
     refresh();
     if ((sockfd = connect_to_gopher(GSgetHost(ZeGopher), GSgetPort(ZeGopher))) <0) {
        check_sock(sockfd, GSgetHost(ZeGopher));
        close(filefd);
        return(1);
     }

     /** Send out the request **/
     writestring(sockfd, GSgetPath(ZeGopher));
     writestring(sockfd, "\r\n");

     Draw_Status("Receiving GIF...");
     refresh();

     /* read and read until we get EOF */
     twirl();
     while ( (cc = recv( sockfd, buf, sizeof(buf), 0 )) > 0 ) {
        /* hope we don't have error while writing - I'm so lazy */
        write( filefd, buf, cc );
        twirl();
     }
     close( filefd ); /* should error check */
     if ( cc < 0 )
        Draw_Status( "Error in file" );
     else
        Draw_Status( "File received ok" );
     socketclose( sockfd );
     view_file(Filename, GSgetTitle(ZeGopher));
     if (unlink(Filename)!=0)
	  fprintf(stderr, "Couldn't unlink!!!\n"), exit(-1);

     return(0);
}

#else 
/*
 * fork off a sound process to siphon the data across the net.
 * So the user can listen to tunage while browsing the directories.
 */

do_sound(ZeGopher)
  GopherStruct *ZeGopher;
{
     int i=0, iLength, sockfd;
     char tmpfilename[_MAX_PATH];
     FILE *tmpfile;
     char inputline[512];
     char sTmp[5];
     BOOLEAN Waitforchld = FALSE;


     /** Okay, it's cool, we can fork off **/

     if (SOUNDCHILD != 0)
	  Waitforchld = TRUE;


     if ( (SOUNDCHILD = fork()) < 0)
	  ;/* Fork Error */
     
     else if (SOUNDCHILD == 0) {  /* Child Process */
	  wait(SIGCHLD);
	  suck_sound(sockfd);
	  exit(0);
     }
     
     /* Parent Process */
     
     socketclose(sockfd);
}
	       

/*
 * this procedure just retrieves binary data from the socket and
 * pumps it into a "play" process.
 */


suck_sound(sockfd)
  int sockfd;
{
     FILE *Play;
     int j;
     char buf[BUFSIZE];

     if (PlayCommand[0] == '\0') {
	  /*** Hey! no play command, bummer ***/
	  CursesErrorMsg("Sorry, this machine doesn't support sounds");

	  return(0);
     }

     Play = popen(PlayCommand, "w");
	  
     
     while(1) {
          j = recv(sockfd, buf, BUFSIZE,0);
	  
	  if (j == 0)
	       break;
	  
	  fwrite(buf, 1, j, Play);
     }
}

#endif /* MESS_DOS */


/*
 * Replace the searched words with backspaces and underline characters.
 */
static char sGBoldoutput[20];  /*** Used for stripping weird stuff from
				    term strings ***/
static int iGposition = 0;     /*** Pointer into the Boldoutput string **/

/*** Used by tputs() ***/

Boldoutchar(c)
  char c;
{
     sGBoldoutput[iGposition++] = c;
}


void
Boldit(inputline, outputline, MungeSearchstr)
  char *inputline, *outputline, *MungeSearchstr;
{
#ifdef IS_MESS_DOS /* Just copy the line from in to out */
     strcpy(outputline,inputline);
#else  /* This whole routine is skipped for MSDOS */
     char words[20][40];  /** A reasonable guess **/
     char c;
     int numchars, lowwordnum, wordcount, i;
     char *cp, *lowword;

     bzero(outputline, 512); /** non portable!!!! ***/
     
     while (isspace(*MungeSearchstr)) /** Strip off spaces **/
	  MungeSearchstr++;
	  
     for (wordcount=0; wordcount<20; wordcount++) {

	  while (isspace(*MungeSearchstr)) /** Strip off spaces **/
	       MungeSearchstr++;
	  
	  numchars = sreadword(MungeSearchstr, words[wordcount], 40);
	  MungeSearchstr += numchars;
	  if (numchars == 0)
	       break;
	  if (strcmp(words[wordcount], "and")==0 ||
	      strcmp(words[wordcount], "or")==0 ||
	      strcmp(words[wordcount], "not")==0) {
	       words[wordcount][0] = '\0';
	       wordcount--;
	  }
     }


     /** Find the first word in the line **/

     while (*inputline!='\0') {
	  lowword = NULL;

	  for (i=0; i< wordcount; i++) {
	       cp = strcasestr(inputline, words[i]);
	       if (cp != NULL)
		    if (cp < lowword || lowword == NULL) {
			 lowword = cp;
			 lowwordnum = i;
		    }
	  }

	  if (lowword == NULL) {
	       strcpy(outputline, inputline);
	       return;
	  }
	  else {
	       strncpy(outputline, inputline, lowword - inputline);
	       outputline += (lowword - inputline);
	       inputline = lowword;
	       
	       iGposition = 0;
	       tputs(sGHighlighton, 1, Boldoutchar);
	       sGBoldoutput[iGposition] = '\0';
	       strcpy(outputline, sGBoldoutput);
	       outputline += strlen(sGBoldoutput);

	       strncpy(outputline, inputline, strlen(words[lowwordnum]));
	       inputline += strlen(words[lowwordnum]);
	       outputline += strlen(words[lowwordnum]);


	       iGposition = 0;
	       tputs(sGHighlightoff, 1, Boldoutchar);
	       sGBoldoutput[iGposition] = '\0';
	       strcpy(outputline, sGBoldoutput);
	       outputline += strlen(sGBoldoutput);

	       strcpy(outputline, sGHighlightoff);
	       outputline += strlen(sGHighlightoff);
	       
	  }
     }
#endif  /* IS_MESS_DOS */
}

int
twirl()
{
     static int twirlnum = 0;
     static char *twirls = "-\\|/";

     addch('\b');
     addch(*(twirls + (twirlnum++ % 4 )));
     refresh();

     return(0);
}


/**
*** Show file takes an open socket connection, writes it to a file
*** and passes it to your favorite pager.
**/

int
showfile(ZeGopher)
  GopherStruct *ZeGopher;
{
     int i=0, iLength, sockfd;
     char tmpfilename[_MAX_PATH];
     FILE *tmpfile;
     char inputline[512];
     char outputline[512];
     char sTmp[5];
     int twirlcount = 0;

     sTmp[0] = '\0';

     if ((sockfd = connect_to_gopher(GSgetHost(ZeGopher), GSgetPort(ZeGopher))) <0) {
	  check_sock(sockfd, GSgetHost(ZeGopher));
	  return(1);
     }

     Draw_Status("Receiving Text...");
     refresh();
     /** Send out the request **/

     writestring(sockfd, GSgetPath(ZeGopher));
     writestring(sockfd, "\r\n");
     /** Check the result of the command.  See if we screwed up... **/

     /** Open a temporary file **/

     sprintf(tmpfilename, "%sgof%d.txt",tmpDirPath, getpid());

     if ((tmpfile = fopen(tmpfilename, "w")) == NULL)
	  fprintf(stderr, "Couldn't make a tmp file!\n"), exit(-1);

     for(;;) {
          twirlcount++;

          if((twirlcount % 25) == 0)
               twirl();

	  iLength = readline(sockfd, inputline, 512);
	  outputline[0] = '\0';
	  if (iLength == 0)
	       break;

	  ZapCRLF(inputline);
	  
	  
	  /*** Ugly hack ahead..... ***/

          if (GSgetType(ZeGopher) == A_CSO) {
	       if (inputline[0] == '2')
		    break;

	       if ((inputline[0] >= '3') && (inputline[0] <= '9'))  {
		    fprintf(tmpfile, "%s\n", GSgetPath(ZeGopher));
		    fprintf(tmpfile, "%s\n", inputline+4);
		    break;
	       }
	       if (inputline[0] == '-') {
		    if (inputline[5] + (inputline[6] == ':' ? 0 : inputline[6]) != i)
			 fprintf(tmpfile, "-------------------------------------------------------\n");
		    i = inputline[5] + (inputline[6] == ':' ? 0 : inputline[6]);
		    fprintf(tmpfile, "%s\n", inputline+7);
	       }
	  }

          if (GSgetType(ZeGopher) == A_FILE || 
	      GSgetType(ZeGopher) == A_EVENT) {
	       if ((inputline[0] == '.') && (inputline[1] == '\0'))
		    break;
	       else {
		    /*** Underline searched words, except and, or and not ***/
		    if (Searchstring != NULL)
			 Boldit(inputline, outputline, Searchstring);
		    else
			 strcpy(outputline, inputline);

		    fprintf(tmpfile, "%s\n", outputline);
	       }
	  } else if (GSgetType(ZeGopher) == A_MIME) {
		if ((inputline[0] == '.') && (inputline[1] == '\0'))
		 break;
		else {
		 fputs(inputline, tmpfile);
		 fputc('\n', tmpfile);
               }
         }
     }

     socketclose(sockfd);
     fprintf(tmpfile, "\012 \n\n");  /** Work around a bug in xterm n' curses*/
     fclose(tmpfile);

     if(GSgetType(ZeGopher) == A_MIME)
	display_mime(tmpfilename, GSgetTitle(ZeGopher));
     else
	display_file(tmpfilename, GSgetTitle(ZeGopher));

     /** Good little clients clean up after themselves..**/

     if (unlink(tmpfilename)!=0)
	  fprintf(stderr, "Couldn't unlink!!!\n"), exit(-1);

     return(0);
}

/**
*** getfile takes an open socket connection, and writes it to a file.
*** Very much line showfile(), except no '.' check, no underlining,
*** no displaying and so on.  And precious little error checking :-(
**/
int 
NEAR getfile(ZeGopher)
  GopherStruct *ZeGopher;
{
     char *p, buf[BUFSIZE];   /* hope this is long enough :-) */
     int cc, sockfd, filefd;

     /* Try to construct a nice default filename to save in */
     p = rindex(GSgetPath(ZeGopher), '/' );
     if ( p && *(p+1) ) {
        strcpy( buf, p+1 );
     } else {
        /* no slashes, leave it empty */
        buf[0] = '\0';
     }
     GetOneOption("Save in file: ",buf);
     if ( buf[0] == '\0' ) {  /* empty name */
        Draw_Status("Retrieval abandoned");
        return(1);
     }
     /* Something there, maybe just spaces.  Oh well. */

     /* open the file - if it already exists, we're about to clobber it */
#ifdef IS_MESS_DOS
     if ( (filefd = open( buf, O_WRONLY|O_TRUNC|O_CREAT|O_BINARY, S_IWRITE )) < 0 ) {
#else
     if ( (filefd = open( buf, O_WRONLY|O_TRUNC|O_CREAT, 0666 )) < 0 ) {
#endif
        /* should give more info here */
        Draw_Status( "Could not open file" );
        return(1);
     }

     Draw_Status("Connecting to server...");
     refresh();
     if ((sockfd = connect_to_gopher(GSgetHost(ZeGopher), GSgetPort(ZeGopher))) <0) {
        check_sock(sockfd, GSgetHost(ZeGopher));
        close(filefd);
        return(1);
     }

     /** Send out the request **/
     writestring(sockfd, GSgetPath(ZeGopher));
     writestring(sockfd, "\r\n");

     Draw_Status("Receiving file...");
     refresh();
     twirl();
     /* read and read until we get EOF */
     while ( (cc = recv( sockfd, buf, sizeof(buf), 0 )) > 0 ) {
        /* hope we don't have error while writing - I'm so lazy */
        write( filefd, buf, cc );
     }
     twirl();
     close( filefd ); /* should error check */
     if ( cc < 0 )
        Draw_Status( "Error in file" );
     else
        Draw_Status( "File received ok" );
     socketclose( sockfd );
     return(0);
}





/*
** Pushgopher takes a GopherThing pointer and adds it to it's stack.
**
** Ick this must be fixed!
*/

int
NEAR pushgopher(ZeDir)
  GopherDirObj *ZeDir;
{

        OldDirs[iLevel]= ZeDir;
        iLevel ++;

}

/*
** If the stack is empty, popgopher returns a -1
*/

int
NEAR popgopher(ZeDir)
  GopherDirObj **ZeDir;
{

     if (iLevel == 0)
	  return(-1);

     iLevel --;

     *ZeDir =  OldDirs[iLevel];

     return(0);
}


void check_sock(sockfd, host)
  int sockfd;
  char *host;
{
     char DispString[WHOLELINE];
     char Response[HALFLINE];

     Response[0] = '\0';

     if (sockfd <0) {
	  sprintf(DispString, "Cannot connect to host %s: ", host);
	  CursesErrorMsg(DispString);
     }
}

#ifdef IS_MESS_DOS
void controlc( int sig)
{
	signal(SIGINT, SIG_IGN);
        exit_curses();
     	exit(3);
}
#endif

/**************
** This bit of code is executed when the program exits/control c is hit.
*/
void
cleanup(void)
{
     char buf[_MAX_PATH];

     sprintf(buf, "%sgof%d.txt", tmpDirPath, getpid());
     if (access(buf,00) != -1){
     	if (unlink(buf) < 0)
	  fprintf(stderr, "could not unlink %s\n", buf);
	}
#ifndef IS_MESS_DOS
     exit(0);
#endif
}

#ifndef IS_MESS_DOS
/**************
** This bit of code catches window size change signals
**/

void*
sizechange()
{
#ifdef  TIOCGWINSZ
     static struct      winsize zewinsize;        /* 4.3 BSD window sizing */
#endif
     
#ifdef  TIOCGWINSZ
     if (ioctl(0, TIOCGWINSZ, (char *) &zewinsize) == 0) {
	  LINES = zewinsize.ws_row;
	  COLS = zewinsize.ws_col;
     } else {
#endif
	  /* code here to use sizes from termcap */
#ifdef  TIOCGWINSZ
     }
     exit_curses();
     MainWindow = initscr();
     init_curses();
     
     scline(iMenuLines, -1, 1, CurrentDir, SavedTitle);

#endif
}
#endif /* IS_MESS_DOS */


/**********
**
** Set up all the global variables.
**
***********/

void
NEAR Initialize()
{
     int err;
     char *cp;
     char termname[HALFLINE];
     static char terminal[1024];
     static char capabilities[1024];   /* String for cursor motion */
     static struct stat statbuf;
     
     static char *ptr = capabilities;  /* for buffering         */

     /** Get the pager command **/
     strcpy(PagerCommand, PAGER_COMMAND);
     strcpy(MailCommand, MAIL_COMMAND);
     strcpy(PlayCommand, PLAY_COMMAND);
     strcpy(TelnetCommand, TELNET_COMMAND);
     strcpy(TN3270Command, TN3270_COMMAND);
     strcpy(MIMECommand, MIME_COMMAND);
#ifdef IS_MESS_DOS
     strcpy(GifCommand, GIF_COMMAND);
#else
     strcpy(PrinterCommand, PRINTER_COMMAND);
#endif

     read_rc();

     if ((cp = getenv("TEMP")) != NULL && (stat(cp , &statbuf)== 0)
     	&& statbuf.st_mode & S_IFDIR) {
          strcpy(tmpDirPath, cp);
          if(tmpDirPath[strlen(tmpDirPath) - 1] != '\\')
		strcat(tmpDirPath, "\\");
     } else
     if ((cp = getenv("TMP")) != NULL && (stat( cp, &statbuf)== 0)
	&& statbuf.st_mode & S_IFDIR) {
		strcpy(tmpDirPath, cp);
                if(tmpDirPath[strlen(tmpDirPath) - 1] != '\\')
			strcat(tmpDirPath, "\\");
	  } else	
		strcpy(tmpDirPath, "");
#ifdef SPAWNO
     init_SPAWNO(tmpDirPath, SWAP_ANY);
#endif
     if (getenv("PAGER") != NULL)
	  strcpy(PagerCommand, getenv("PAGER"));

     if(getenv("MAILER") != NULL)
         strcpy(MailCommand, getenv("MAILER"));

     if(getenv("TELNET") != NULL)
          strcpy(TelnetCommand, getenv("TELNET"));

     if(getenv("TN3270") != NULL)
	  strcpy(TN3270Command, getenv("TN3270"));

     if (getenv("PLAYER") != NULL)
          strcpy(PlayCommand, getenv("PLAYER"));

     if (getenv("MIMECMD") != NULL)
	  strcpy(MIMECommand, getenv("MIMECMD"));

#ifdef IS_MESS_DOS
     if(getenv("VIEWER") != NULL)
         strcpy(GifCommand, getenv("VIEWER"));
#endif
     if (getenv("EDITOR") == NULL)
	  strcpy(EditorCommand, "vi ");
     else
	  strcpy(EditorCommand, getenv("EDITOR"));

#ifdef IS_MESS_DOS /* Don't have to deal with termcap stuff in DOS */ 
          strcpy(sGAudibleBell, "\007");
          strcpy(sGClearscreen, "");
#else
     /*** get the terminal type ***/

     if (getenv("TERM") == null)
	  fprintf(stderr, "I don't understand your terminal type\n"), exit(-1);
     
     if (strcpy(termname, getenv("TERM")) == null)
	  fprintf(stderr, "I don't understand your terminal type\n"),exit(-1);
     
     if ((err = tgetent(terminal, termname)) != 1)
	  fprintf(stderr, "I don't understand %s terminals\n",termname),exit(-1);
     
     if ((cp = (char *)tgetstr("cl", &ptr)) != null)
	  strcpy(sGClearscreen, cp);
     else
	  strcpy(sGClearscreen, "");

     if ((cp = (char *) tgetstr("bl", &ptr)) != null)
	  strcpy(sGAudibleBell, cp);
     else
	  strcpy(sGAudibleBell,"\007");

     if ((cp = (char *) tgetstr("so", &ptr)) != null) {
	  strcpy(sGHighlighton, cp);
	  if ((cp = (char *) tgetstr("se", &ptr)) != null) {
	       strcpy(sGHighlightoff, cp);
	  }
	       
     }

     else {
	  fprintf(stderr, "Warning, this terminal has no highlight mode!\n");
	  sGHighlighton[0] = '\0';
	  sGHighlightoff[0] = '\0';
     }

     /*** init mainwindow ****/
     tputs(sGClearscreen,1, outchar);     
#endif /* MESS_DOS */
     init_curses();
     MainWindow = stdscr;

     /*** make a signal handler for window size changes ***/

#ifdef IS_MESS_DOS
    atexit(cleanup);
    signal(SIGINT, controlc);
#else	  /* except that dos can't do sigwinch */


#ifdef SIGWINCH
     if (signal(SIGWINCH, sizechange)==(void*)-1)
	  perror("signal died:\n"), exit(-1);

#endif

     if (signal(SIGINT, cleanup) == (void*)-1)
	  perror("signal died:\n"), exit(-1);

#endif /* MESS_DOS */

}

#ifdef IS_MESS_DOS
/* No simple way to print in DOS.  They can always save to PRN */
static char *GlobalOptions[] =  
{"Pager Command", "Telnet Command", "3270 Emulator Command", "Sound Command",
	"Mail Command", "MIME Pager", "GIF Command"};
#else
static char *GlobalOptions[] =  
{"Pager Command", "Telnet Command", "3270 Emulator Command", "Sound Command",
	"Mail Command", "MIME Pager", "Print Command"};
#endif


void
NEAR SetOptions()
{
     char ch;
     static char Response[MAXRESP][MAXSTR];
     
     strcpy(Response[0], PagerCommand);
     strcpy(Response[1], TelnetCommand);
     strcpy(Response[2], TN3270Command);
     strcpy(Response[3], PlayCommand);
     strcpy(Response[4], MailCommand);
     strcpy(Response[5], MIMECommand);
#ifdef IS_MESS_DOS
     strcpy(Response[6], GifCommand);
#else
     strcpy(Response[6], PrinterCommand);
#endif

     if (SecureMode) {
	  CursesErrorMsg("Sorry, you are not allowed to set options in secure mode.");
	  return;
     }

     Get_Options("Internet Gopher Options", "", 7, GlobalOptions, Response);

     strcpy(PagerCommand, Response[0]);
     strcpy(TelnetCommand, Response[1]);
     strcpy(TN3270Command, Response[2]);
     strcpy(PlayCommand, Response[3]);
     strcpy(MailCommand, Response[4]);
     strcpy(MIMECommand, Response[5]);
#ifdef IS_MESS_DOS
     strcpy(GifCommand, Response[6]);
#else
     strcpy(PrinterCommand, Response[6]);
#endif
     ChangedDefs = TRUE;
}



/* This should be a generalized stack type.  This is icky for now... */
static int SavedLinenum[MAXGOPHERS];
static int SavedLinePtr = 0;
static char TitleStack[OD_MAX][WHOLELINE];
static int TopTitle = 0;

main(argc, argv)
  int argc;
  char *argv[];
{
     int iLine=0;
     int iNum=0;
     BOOLEAN bDone = FALSE;
     char sTmp[80];
     GopherStruct *TmpGopher, *Gopherp;
     char TypedChar;
     /*** for getopt processing ***/
     int c;
     extern char *optarg;
     extern int optind;
     int errflag =0;
     BOOLEAN Garbled = TRUE;
     BOOLEAN InitialBookmarks = FALSE;
#undef BETA_BUILD
#ifdef BETA_BUILD
/* sec, min, hour, mday, mon, year, all 0-based */
     struct tm expires = { 0, 0, 0, 1, 8, 93, };
     time_t now;
#endif
#ifdef IS_MESS_DOS
     char *dos_path, *foo;
     void *s_malloc();
     char HelpPath[_MAX_PATH], drive[_MAX_DRIVE], dir[_MAX_DIR];

#ifdef PCTCP
/* A design decision made when UT Med school people first started using
   FTP Software's PC/TCP left the path where all the software is installed
   out of the path and relied on a slew of batch files to do the work.
   <sigh> this won't work any more
*/
     sock_init();
//     tcp_set_debug_state(1);

     foo = getenv("PATH");
     dos_path = (char *) s_malloc(strlen(foo) + strlen(PCTCP_DIR) + 10);
     if (dos_path == NULL) {
	 fprintf(stderr, "%s: out of memory\n", argv[0]);
	 exit(255);
     }
     sprintf(dos_path, "PATH=%s;%s", foo, PCTCP_DIR);
     if(putenv(dos_path) == -1)
	 fprintf(stderr, "%s: Couldn't change path\n", argv[0]);
#endif
/* Save the path the executable is in. Generate path to the help file.
   I think the argv[0] thing only works with DOS 3.0 for later */
    _splitpath(argv[0], drive, dir, NULL, NULL);
    sprintf(InstallPath, "%s%s", drive, dir);
    sprintf(HelpPath, "%s%s", InstallPath, GOPHERHELP);

#endif /* IS_MESS_DOS */
#ifdef BETA_BUILD
     time(&now);
     if(now > mktime(&expires)) {
	printf("This is a beta-test build of ugopher and has expired.\n");
	printf("Please obtain the latest version from oac.hsc.uth.tmc.edu\n");
	exit(0);
     }
#endif
     TmpGopher = GSnew();

     SavedLinenum[SavedLinePtr] = 1;

     GSsetPort (TmpGopher,  GOPHER_PORT);
     GSsetType (TmpGopher, A_DIRECTORY);
//     GSsetType (TmpGopher, 1);
     GSsetPath (TmpGopher,"");
     GSsetHost (TmpGopher, DEFAULT_HOST);
     sTmp[0] = '\0';

     while ((c = getopt(argc, argv, "bsp:t:")) != -1)
	  switch (c) {
	  case 's':
	       SecureMode = TRUE;
	       break;
	  case 'p':
	       GSsetPath(TmpGopher, optarg);
	       break;
	  case 't':
	       GSsetTitle(TmpGopher, optarg);
	       break;
	  case 'b':
	       InitialBookmarks = TRUE;
	       break;
	  case '?':
	       errflag++;
	  }

     if(GSgetTitle(TmpGopher) == NULL) {
	  sprintf(sTmp, "Root gopher server: %s", GSgetHost(TmpGopher));
	  GSsetTitle(TmpGopher, sTmp);
     }
     strcpy(TitleStack[TopTitle], GSgetTitle(TmpGopher));

     if (errflag) {
	  fprintf(stderr, "Usage: %s [-s] [-p path] [-t title] [hostname port]\n", argv[0]);
	  exit(-1);
     }

     if (optind < argc) {
	  GSsetHost(TmpGopher, argv[optind]);
	  optind++;
     }

     if (optind < argc) {
	  GSsetPort(TmpGopher, atoi(argv[optind]));
	  optind++;
     }
#ifdef PCNFS
	if (loaded() != 1) {
		fprintf(stderr, "%s: PC-NFS is not loaded.", argv[0]);
		exit(-1);
	}
#endif
#ifdef LWP
	if (loaded() != 1) {
		fprintf(stderr, "%s: Lan WorkPlace is not loaded.", argv[0]);
		exit(-1);
	}
#endif
#ifdef PCTCP
	/* Prints error message and calls exit() if Kernel not loaded */
	_find_vec();
#endif

     /*** Set up global variables, etc. ***/

     Initialize();

     iLine = 1;

//     printf("host: %s port:%i type:%i path:%s",GSgetHost(TmpGopher), GSgetPort(TmpGopher), GSgetType(TmpGopher), GSgetPath(TmpGopher));
//     sleep(15);

     if (InitialBookmarks) {
	CurrentDir = GDBookmarks;
	if (CurrentDir != NULL)
	   GDaddGS(CurrentDir, TmpGopher);
	iMenuLines = GDgetTop(CurrentDir);
	strcpy(SavedTitle, "Bookmarks");
	strcpy(TitleStack[TopTitle], "Bookmarks");
     } else
	   process_request(TmpGopher);

     if (iMenuLines <= 0) {
	  /*
	   * We didn't get anything from that gopher server.  Either
	   * it is down, doesn't exist, or is empty or otherwise
	   * busted.
	   */

	  exit_curses();
	  fprintf(stderr,
		  "%s: Nothing received for main menu, can't continue\n", argv[0]);
	  exit(1);
     }

     while (bDone == FALSE)
     {
	  iLine = GetMenu(iMenuLines, CurrentDir, SavedTitle, &TypedChar, iLine, Garbled);
	  Garbled = TRUE;

	  switch(TypedChar)
	  {
	  case '\r':
	  case '\n':
	       /*** Select the designated item ***/
	       iNum = iLine - 1;
	       if (GSgetType(GDgetEntry(CurrentDir, iNum)) == A_DIRECTORY ||
		   GSgetType(GDgetEntry(CurrentDir, iNum)) == A_INDEX) {
		    SavedLinenum[++SavedLinePtr] = iLine;
		    iLine=1;
		    strcpy(TitleStack[++TopTitle],
			   GSgetTitle(GDgetEntry(CurrentDir, iNum)));
	       }
	       if (process_request(GDgetEntry(CurrentDir, iNum))==1) {
		    iLine = SavedLinenum[(SavedLinePtr ==0) ? 0:SavedLinePtr--];
		    if(TopTitle > 0)
			 --TopTitle;
	       }
	       break;

	  case '\0':
	       /*** What the heck? ***/
	       CursesErrorMsg("Strange Error occurred!");
	       break;

	  case 'u':
	  case 'U':
	  {
	       GopherDirObj *tempGdir;

	       /*** Go up a directory level ***/
	       iNum=0;
	       tempGdir = CurrentDir;
	       if ((popgopher(&CurrentDir) == 0) && tempGdir != CurrentDir)
		    if (tempGdir != GDBookmarks)
		       GDdestroy(tempGdir);
	       iLine = SavedLinenum[(SavedLinePtr ==0) ? 0:SavedLinePtr--];
	       strcpy(SavedTitle, TitleStack[(TopTitle == 0) ? 0:--TopTitle]);
	       iMenuLines = GDgetTop(CurrentDir);
	  }
	       break;

	  case 'G':
	      debugStatus = !debugStatus;
	      break;
	  case 'B':
	      CursesErrorMsg("Build stamp: " __DATE__ " at " __TIME__);
	      break;
	  case 's':
	       /*** Save the thing in a file ***/
	       break;

	  case 'M':
	  case 'm':
	  {
	       GopherDirObj *tempGdir = NULL;
	       iNum = 0;
	       while (popgopher(&tempGdir) != -1) {
		    if(tempGdir != NULL) {
			 GDdestroy(CurrentDir);
			 CurrentDir = tempGdir;
		    }
	       }
	       SavedLinePtr = 0;
	       iLine = SavedLinenum[SavedLinePtr];
	       TopTitle = 0;
	       strcpy(SavedTitle, TitleStack[TopTitle]);
	  }

	       iMenuLines = GDgetTop(CurrentDir);
	       break;

	  case 'q':
	  case 'Q':
	       /*** Quit the program ***/
	       bDone = TRUE;
	       exit_curses();
	       break;

	  case 'O':
	       /*** Change various program things ***/
	       SetOptions();
	       break;

	  case '=':
	       iNum = iLine - 1;
	       describe_gopher(GDgetEntry(CurrentDir, iNum));
	       break;

	  case '?':
	       /*** Display help file ***/
#ifdef IS_MESS_DOS
	       display_file(HelpPath, "Gopher Help File");
#else /* Not DOS */
	       display_file(GOPHERHELP, "Gopher Help File");
#endif /* IS_MESS_DOS */
	       break;
	  case '!':
		if (!SecureMode) {
			exit_curses();
#ifdef IS_MESS_DOS
			system(getenv("COMSPEC"));
#else
			system(getenv("SHELL"));
#endif
			init_curses();
		} else {
			printf("%s", sGAudibleBell);
			Garbled = FALSE;
		}
		break;

          case 'v':
	       if(GDBookmarks == NULL) {
		 CursesErrorMsg("No bookmarks are defined");
		 break;
               }

	       SavedLinenum[++SavedLinePtr] = iLine;
	       iLine=1;

	       if (CurrentDir != NULL) {
		pushgopher(CurrentDir);
                strcpy(TitleStack[++TopTitle], "Bookmarks");
	       }

	       CurrentDir = GDBookmarks;
	       iMenuLines = GDgetTop(CurrentDir);
	       strcpy(SavedTitle, "Bookmarks");

	       break;

	  case 'a': /* Add current item as a bookmark */
	  {
	       GopherObj *tmpgs;
	       char newtitle[256];
	       
	       if (GDBookmarks == NULL)
 		    GDBookmarks = GDnew(OD_MAX);

	       tmpgs = GSnew();
	       GScpy(tmpgs, GDgetEntry(CurrentDir, iLine-1));
	       
	       strcpy(newtitle, GSgetTitle(tmpgs));
	       GetOneOption("Name for this bookmark? ", newtitle);
	       if (*newtitle == '\0')
		    break;

	       GSsetTitle(tmpgs, newtitle);
	       GDaddGS(GDBookmarks, tmpgs);
	       GSdestroy(tmpgs);

	       ChangedDefs = TRUE;

	       break;
	  }

	  case 'A': /* Add the current directory or search as a bookmark */
	  {
		GopherObj *TmpGS;
		char tTitle[256];

		if (GDBookmarks == NULL)
		     GDBookmarks = GDnew(OD_MAX);

		if (CurrentDir == GDBookmarks) {
		     CursesErrorMsg("Can't create a bookmarks of your bookmarks");
		     break;
		}

		TmpGS = GSnew();
		if (iLevel == 0)
		     GScpy(TmpGS, TmpGopher);
		else
		    GScpy(TmpGS, GDgetEntry(OldDirs[iLevel-1], SavedLinenum[SavedLinePtr]-1));

		strcpy(tTitle, TitleStack[TopTitle]);
		GetOneOption("Name for this bookmark? ", tTitle);
		if (*tTitle == '\0')
		     break;

		GSsetTitle(TmpGS, tTitle);

		/* Special handling for searches */
		if (GSgetType(TmpGS) == A_INDEX) {
		     char SearchPath[512];
		     strcpy(SearchPath, GSgetPath(TmpGS));
		     strcat(SearchPath, Searchstring);
		     GSsetPath(TmpGS, SearchPath);
		     GSsetType(TmpGS, A_DIRECTORY);
		}

		GDaddGS(GDBookmarks, TmpGS);
		ChangedDefs = TRUE;
		break;

	  }

	  case 'd': /* Delete Current Bookmark item */
	  {
	      GopherDirObj *TempGD;
	      int i;

	      if (CurrentDir != GDBookmarks) {
                printf("%s", sGAudibleBell);
		Garbled = FALSE;
		break;
	      }

	      if (GDgetTop(CurrentDir) == 1) {
		iNum = 0;
		TempGD = CurrentDir;
	        if ((popgopher(&CurrentDir) == 0) && TempGD != CurrentDir) {
		    if (TempGD != GDBookmarks)
	            	GDdestroy(TempGD);
		    iLine = SavedLinenum[(SavedLinePtr ==0) ? 0:SavedLinePtr--];
                    strcpy(SavedTitle, TitleStack[(TopTitle == 0) ? 0:--TopTitle]);
		    iMenuLines = GDgetTop(CurrentDir);
	        } else 
		    CursesErrorMsg("Can't delete the entire top level directory");
                ChangedDefs = TRUE;
	     }
	      

	    TempGD = GDnew(GDgetTop(CurrentDir) + 1);

	    for (i = 0; i < GDgetTop(CurrentDir); i++) {
		if (i == 0 || i != (iLine - 1))
			GDaddGS(TempGD, GDgetEntry(CurrentDir, i));
		else
			iLine--;
            }

	    GDdestroy(CurrentDir);

	    if (CurrentDir == GDBookmarks)
	    	GDBookmarks = TempGD;

	    CurrentDir = TempGD;
	    ChangedDefs = TRUE;
	    iMenuLines = GDgetTop(CurrentDir);

            break;
	  }
    
	  default :
               printf("%s", sGAudibleBell);
               Garbled = FALSE;
	       break;

	  }
     }
     GSdestroy(TmpGopher);
     if(ChangedDefs)
	write_rc();

     exit(0);
}     


int
NEAR Load_Index(ZeGopher)
  GopherStruct *ZeGopher;
{
     Draw_Status("Searching Text...");
     refresh();

     return(Load_Index_or_Dir(ZeGopher, Searchstring));
}

int
NEAR Load_Dir(ZeGopher)
  GopherStruct *ZeGopher;
{
     Searchstring= NULL;
     return(Load_Index_or_Dir(ZeGopher, NULL));
}


int
NEAR Load_Index_or_Dir(ZeGopher, Searchmungestr)
  GopherStruct *ZeGopher;
  char *Searchmungestr;
{
     int failed = 0;
     int sockfd;
     int i, length;
     char sTmp[10];
     char sOldTitle[WHOLELINE];
     static char inputline[MAXLINE];
     GopherDirObj *NewDir = NULL;

     NewDir = GDnew(INITIAL_GOPHERS);

     sTmp[0]= '\0';

     strcpy(sOldTitle, SavedTitle);

     if ((sockfd = connect_to_gopher(GSgetHost(ZeGopher), GSgetPort(ZeGopher))) <0) {
	  check_sock(sockfd, GSgetHost(ZeGopher));
	  GDdestroy(NewDir);
	  failed = 1;
     }
     else {
          Draw_Status("Receiving Directory...");
          refresh();
	  if (GSgetType(ZeGopher) == A_DIRECTORY) {
	       writestring(sockfd, GSgetPath(ZeGopher));
	       writestring(sockfd, "\r\n");
	  }
	  else if (GSgetType(ZeGopher) == A_INDEX) {
	       writestring(sockfd, GSgetPath(ZeGopher));
	       writestring(sockfd, "\t");
	       writestring(sockfd, Searchmungestr);
	       writestring(sockfd, "\r\n");
	  }

	  if (GSgetType(ZeGopher) == A_INDEX)
	     sprintf(SavedTitle, "%s: %s", GSgetTitle(ZeGopher), Searchmungestr);
          else
	     strncpy(SavedTitle, GSgetTitle(ZeGopher), WHOLELINE);

          SavedTitle[WHOLELINE-1] = '\0';

	  /** Don't push an empty gopher directory... **/
	  if (CurrentDir != NULL)
	       pushgopher(CurrentDir);

	  CurrentDir = NewDir;
	  
	  i = GDfromNet(NewDir, sockfd);

	  if (i <= 0) {
	       CursesErrorMsg("Nothing available.");
	       popgopher(&CurrentDir);
	       strcpy(SavedTitle, sOldTitle);
	       failed = 1;
	       return(failed);
	  }
	  else
	       iMenuLines = i;
	  
     }
     if(sockfd > -1)
         i = socketclose(sockfd);
     return(failed);
}



int
NEAR process_request(ZeGopher)
  GopherStruct *ZeGopher;
{
     int failed=0;
     int sockfd;
     char *cp;

     switch(GSgetType(ZeGopher)) {
     case -1:
	  break;

     case A_EVENT:
/*	  HandleEvent(ZeGopher);*/
	  break;

     case A_FILE:
     case A_MIME:
	  Draw_Status("Connecting to Host...");
	  refresh();
	  failed = showfile(ZeGopher);
	  break;

     case A_MACHEX:
     case A_PCHEX:
     case A_TARORZ:
	  Draw_Status("Connecting to Host...");
	  refresh();
	  if (!SecureMode)
	       failed = getfile(ZeGopher);
	  else {
	       CursesErrorMsg("Sorry, cannot transfer files in securemode");
	       failed = 1;
	  }
	  break;

     case A_DIRECTORY:
	  Draw_Status("Connecting to Host...");
	  refresh();
	  failed = Load_Dir(ZeGopher);
	  break;

     case A_TELNET:
     case A_TN3270:
	  failed = do_telnet(ZeGopher);
	  break;

     case A_INDEX:
	  Draw_Status("Searching Text...");
	  refresh();
	  Searchstring = do_index(ZeGopher);
	  if (Searchstring != NULL)
	       failed=Load_Index(ZeGopher);
	  else
	       failed = 1;
	  break;

     case A_CSO:
	  failed = do_cso(ZeGopher);
	  break;

     case A_SOUND:
	  Draw_Status("Receiving Sound...");
	  refresh();
	  failed = do_sound(ZeGopher);
	  break;

#ifdef IS_MESS_DOS
     case A_GIF:
#endif
     case A_IMAGE:
	  Draw_Status("Receiving Image...");
	  refresh();
	  failed = do_gif(ZeGopher);
	  break;

     }
     return(failed);
}


int
NEAR describe_gopher(ZeGopher)
  GopherStruct *ZeGopher;
{
     char tmpfilename[_MAX_PATH];
     FILE *tmpfile;

     sprintf(tmpfilename,"%sgof%d.txt", tmpDirPath, getpid());
     if ((tmpfile = fopen(tmpfilename, "w")) == NULL)
	   fprintf(stderr, "Couldn't make a tmp file!\n"), exit(-1);

     fprintf(tmpfile,"Name=%s\nType=%c\nPort=%d\nPath=%s\nHost=%s\n\n",
	     GSgetTitle(ZeGopher),
	     GSgetType(ZeGopher),
	     GSgetPort(ZeGopher),
	     GSgetPath(ZeGopher),
	     GSgetHost(ZeGopher));

     fclose(tmpfile);

     display_file(tmpfilename, "Gopher Link Data");
     if (unlink(tmpfilename) != 0)
	  fprintf(stderr, "Couldn't unlink!!!\n"), exit_curses(), exit(-1);

	  
     return(0);	 
}
