
/*
 * Network Utilities for MS-DOS TCP/IP Stacks to support the
 * Gopher client under MS-DOS
 * By Stan Barber, Baylor College of Medicine
 * This file is Copyright, 1992, Stan Barber and Baylor College of Medicine.
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use the contents of this file as long as: there is no monetary profit gained
 * specifically from the use or reproduction of the contents of this file, it is
 * not sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 * The authors make no claims as to the fitness or correctness of this
 * software for any use whatsoever, and it is provided as is. Any use of the
 * software is at the user's own risk. 
 */
#include "gopher.h"

#ifdef PCNFS
#include <dos.h>
/* See if PC-NFS is loaded. Works like the LanWorkplace call. */
#define IOCTL_FUNC		0x44
#define IOCTL_SPECIAL_BASE	0x0E0
#define IOCTL_READ_8		IOCTL_SPECIAL_BASE + 0
#define X_GET_NFS_INFO		20
/*
 * Extended IOCTL parameter block structure
 */
 
struct ipb_struc {
	unsigned char	ipb_key;		/* key for X_ioctl calls */
	unsigned char	ipb_func;		/* extended func number */
	unsigned int	ipb_offset;		/* offset for data */
	unsigned int 	ipb_segment;		/* segment for data */
	unsigned int	ipb_count;
	unsigned int	ipb_pointer;
	unsigned int	ipb_nat_index;
};

int
loaded(void)
{
	/*
	 * This routine uses the X_GET_NFS_INFO ioctl to see if PC-NFS 
	 * has been installed.
	 */
	 
	struct ipb_struc ipb;
	union REGS	inregs, outregs;
	struct SREGS segregs;
	char far *p;
	
	inregs.h.ah = IOCTL_FUNC;
	inregs.h.al = IOCTL_READ_8;
	ipb.ipb_func = X_GET_NFS_INFO;
	
	segread(&segregs);
	p = (char far *) &ipb;
	inregs.x.dx = FP_OFF (p);
	segregs.ds = FP_SEG (p);
	intdosx (&inregs, &outregs, &segregs);
	return(!(outregs.x.cflag));
}	
#endif
#ifdef LWP
/* 
 * special version of gethostbyname() for Lan Workplace for DOS
 * Why the LWP gethostbyname() function does not work this way, I don't know. 
 */
#include <arpa/nameser.h>
#include <resolv.h>

#define	MAXALIASES	35
#define	MAXADDRS	35

static char *h_addr_ptrs[MAXADDRS + 1];
static struct hostent hostval;
static char *host_aliases[MAXALIASES];
static char hostbuf[BUFSIZ+1];
static struct in_addr host_addr;
static char hostaddr[MAXADDRS];
static char *host_addrs[2];

#if PACKETSZ > 1024
#define	MAXPACKET	PACKETSZ
#else
#define	MAXPACKET	1024
#endif

typedef union {
    HEADER hdr;
    unsigned char buf[MAXPACKET];
} querybuf;

typedef union {
    long al;
    char ac;
} align;


/*
 * Skip over a compressed domain name. Return the size or -1.
 */
__dn_skipname(comp_dn, eom)
	const unsigned char *comp_dn, *eom;
{
	register unsigned char *cp;
	register int n;

	cp = (unsigned char *)comp_dn;
	while (cp < eom && (n = *cp++)) {
		/*
		 * check for indirection
		 */
		switch (n & INDIR_MASK) {
		case 0:		/* normal case, n == len */
			cp += n;
			continue;
		default:	/* illegal type */
			return (-1);
		case INDIR_MASK:	/* indirection */
			cp++;
		}
		break;
	}
	return (cp - comp_dn);
}

static struct hostent *
getanswer(answer, anslen, iquery)
	querybuf *answer;
	int anslen;
	int iquery;
{
	register HEADER *hp;
	register unsigned char *cp;
	register int n;
	struct hostent *whp;
	unsigned char *eom;
	char *bp, **ap;
	int type, class, buflen, ancount, qdcount;
	int haveanswer, getclass = C_ANY;
	char **hap;
	
	eom = answer->buf + anslen;
	/*
	 * find first satisfactory answer
	 */
	hp = &answer->hdr;
	ancount = ntohs(hp->ancount);
	qdcount = ntohs(hp->qdcount);
	bp = hostbuf;
	buflen = sizeof(hostbuf);
	cp = answer->buf + sizeof(HEADER);
	if (qdcount) {
		if (iquery) {
			if ((n = dn_expand((unsigned char *)answer->buf,
			    (unsigned char *)eom, (unsigned char *)cp,
			    (unsigned char *)bp, buflen)) < 0) {
				return ((struct hostent *) NULL);
			}
			cp += n + QFIXEDSZ;
			hostval.h_name = bp;
			n = strlen(bp) + 1;
			bp += n;
			buflen -= n;
		} else
			cp += __dn_skipname(cp, eom) + QFIXEDSZ;
		while (--qdcount > 0)
			cp += __dn_skipname(cp, eom) + QFIXEDSZ;
	} else if (iquery) {
		return ((struct hostent *) NULL);
	}
	ap = host_aliases;
	*ap = NULL;
	hostval.h_aliases = host_aliases;
	hap = h_addr_ptrs;
	*hap = NULL;
	hostval.h_addr_list = h_addr_ptrs;
	haveanswer = 0;
	while (--ancount >= 0 && cp < eom) {
		if ((n = dn_expand((unsigned char *)answer->buf, 
			 (unsigned char *)eom,(unsigned char *)cp,
			 (unsigned char *)bp, buflen)) < 0)
			break;
		cp += n;
		type = _getshort(cp);
 		cp += sizeof(unsigned short);
		class = _getshort(cp);
 		cp += sizeof(unsigned short) + sizeof(unsigned long);
		n = _getshort(cp);
		cp += sizeof(unsigned short);
		if (type == T_CNAME) {
			cp += n;
			if (ap >= &host_aliases[MAXALIASES-1])
				continue;
			*ap++ = bp;
			n = strlen(bp) + 1;
			bp += n;
			buflen -= n;
			continue;
		}
		if (iquery && type == T_PTR) {
			if ((n = dn_expand((unsigned char *)answer->buf,
			    (unsigned char *)eom, (unsigned char *)cp,
			    (unsigned char *)bp, buflen)) < 0) {
				cp += n;
				continue;
			}
			cp += n;
			hostval.h_name = bp;
			return(&hostval);
		}
		if (iquery || type != T_A)  {
			cp += n;
			continue;
		}
		if (haveanswer) {
			if (n != hostval.h_length) {
				cp += n;
				continue;
			}
			if (class != getclass) {
				cp += n;
				continue;
			}
		} else {
			hostval.h_length = n;
			getclass = class;
			hostval.h_addrtype = (class == C_IN) ? AF_INET : AF_UNSPEC;
			if (!iquery) {
				hostval.h_name = bp;
				bp += strlen(bp) + 1;
			}
		}

		bp += sizeof(align) - ((unsigned long)bp % sizeof(align));

		if (bp + n >= &hostbuf[sizeof(hostbuf)]) {
			break;
		}
		bcopy(cp, *hap++ = bp, n);
		bp +=n;
		cp += n;
		haveanswer++;
	}
	if (haveanswer) {
		*ap = NULL;
		*hap = NULL;
		return (&hostval);
	} else {
		return ((struct hostent *) NULL);
	}
}

struct hostent *
lwpgethostbyname(name)
	char *name;
{
	char workbuf[MAXPACKET];
	querybuf buf;
	register char *cp;
	struct hostent *hp;
	int n;
	extern int dns_errno;
	
	/*
	 * disallow names consisting only of digits/dots, unless
	 * they end in a dot.
	 */
	if (isdigit(name[0]))
		for (cp = name;; ++cp) {
			if (!*cp) {
				if (*--cp == '.')
					break;
				/*
				 * All-numeric, no dot at the end.
				 * Fake up a hostent as if we'd actually
				 * done a lookup.  What if someone types
				 * 255.255.255.255?  The test below will
				 * succeed spuriously... ???
				 */
				if ((host_addr.s_addr = inet_addr(name)) == -1) {
					return((struct hostent *) NULL);
				}
				hostval.h_name = name;
				hostval.h_aliases = host_aliases;
				host_aliases[0] = NULL;
				hostval.h_addrtype = AF_INET;
				hostval.h_length = sizeof(unsigned long);
				h_addr_ptrs[0] = (char *)&host_addr;
				h_addr_ptrs[1] = (char *)0;
				hostval.h_addr_list = h_addr_ptrs;
				return (&hostval);
			}
			if (!isdigit(*cp) && *cp != '.') 
				break;
		}
	hp = gethostbyname(name);
	res_init();
	if (dns_errno) return(hp); 
	n = res_mkquery(QUERY,name,C_IN,T_A, (char *)NULL, 0, NULL,workbuf,
		sizeof(workbuf));
	if (n <= 0) return(hp);	 /* can't build a query */
	if ((n = res_send(workbuf, n, buf.buf, sizeof(buf))) < 0) {
			return (hp);
	}
	return(getanswer(&buf, n, 0));
	
}

#endif


